# Getting Started with the Java Code Modernizer Agent

_A comprehensive tutorial for building AI-powered code modernization tools with the Embabel framework_

---

## Introduction

Welcome to the world of AI-powered code modernization! This tutorial will guide you through setting up and using the Java Code Modernizer Agent - a sophisticated tool that automatically identifies and fixes legacy code patterns in your Java projects.

Whether you're dealing with legacy Java 8 codebases that need upgrading to Java 17+, or Spring Boot applications that require modernization to version 3.5.6, this agent has you covered.

## What You'll Learn

By the end of this tutorial, you'll be able to:

- Set up the Java Code Modernizer Agent
- Configure AI models for code analysis
- Run automated code modernization workflows
- Understand the agent's Git-based safety mechanisms
- Customize the agent for your specific needs

## Prerequisites

Before we begin, make sure you have:

- **Java 21** installed
- **Maven 3.6+** for dependency management
- **Git** for version control
- An **AI model provider** account (OpenAI, Anthropic, or AWS Bedrock)
- Basic knowledge of **Spring Boot** and **Java development**

## Step 1: Project Setup

### Clone and Initialize

First, let's get the project up and running:

```bash
# Clone the repository
git clone <your-modernizer-agent-repo>
cd modernizer-agent

# Verify Java version
java --version
# Should show Java 21

# Build the project
mvn clean compile
```

### Project Structure Overview

Here's what you'll find in the project:

```
modernizer-agent/
├── src/main/java/com/embabel/
│   ├── ProjectNameApplication.java     # Main Spring Boot application
│   └── modernizer/agent/
│       ├── ModernizerAgent.java        # Core agent logic
│       ├── Classification.java         # Migration categories
│       └── Domain.java                 # Data models
├── src/main/resources/
│   ├── application.properties          # Configuration
│   └── prompts/                        # AI prompt templates
│       ├── find_migration_points.jinja
│       └── fix_migration_point.jinja
└── pom.xml                            # Maven dependencies
```

## Step 2: Configuration

### Configure Your AI Provider

Open `src/main/resources/application.properties` and configure your preferred AI model:

#### Option A: Anthropic Claude (Recommended)

```properties
embabel.models.default-llm=us.anthropic.claude-3-5-sonnet-20240620-v1:0
embabel.agent.platform.ranking.llm=us.anthropic.claude-3-5-sonnet-20240620-v1:0
spring.ai.bedrock.anthropic.chat.inference-profile-id=us.anthropic.claude-3-5-sonnet-20240620-v1:0
spring.profiles.active=starwars,bedrock
```

#### Option B: OpenAI GPT

```properties
embabel.models.default-llm=gpt-4
spring.ai.openai.api-key=${OPENAI_API_KEY}
spring.profiles.active=starwars,openai
```

#### Option C: Local Models (Ollama)

```properties
embabel.models.default-llm=llama3.1:8b
embabel.models.default-embedding-model=nomic-embed-text:latest
spring.profiles.active=starwars,ollama
```

### Set Environment Variables

Create a `.env` file or set environment variables:

```bash
# For OpenAI
export OPENAI_API_KEY=your_openai_api_key

# For AWS Bedrock
export AWS_ACCESS_KEY_ID=your_aws_access_key
export AWS_SECRET_ACCESS_KEY=your_aws_secret_key
export AWS_REGION=us-east-1

# For Anthropic
export ANTHROPIC_API_KEY=your_anthropic_api_key
```

## Step 3: Understanding the Agent

### Core Components

The modernizer agent consists of several key components:

#### 1. **ModernizerAgent** - The Brain

This is where the magic happens. The agent:

- Analyzes your codebase for modernization opportunities
- Creates safe Git branches for each change
- Uses AI to generate and apply fixes
- Automatically rolls back failed changes

#### 2. **Classification System**

The agent categorizes issues into four types:

- **Legacy**: Outdated code patterns
- **Deprecated**: Unsupported libraries/frameworks
- **Persistence**: Database-related code
- **Security**: Security-related improvements

#### 3. **Domain Model**

The agent uses a well-structured domain model to represent migration tasks and results:

```mermaid
classDiagram
    class Domain {
        <<abstract>>
    }

    class MigrationTask {
        +String root
        +String notes
        +List~Classification~ classifications
        +MigrationTask(String root, String notes)
        +MigrationTask(String root, String notes, List~Classification~ classifications)
        +SoftwareProject softwareProject()
    }

    class Classification {
        +String name
        +String description
    }

    class MigrationPossibility {
        +String filePath
    }

    class MigrationPoint {
        +String filePath
        +String description
        +String classificationName
    }

    class MigrationPoints {
        +List~MigrationPoint~ migrationPoints
        +String overview
    }

    class MigrationReport {
        +MigrationPoint migrationPoint
        +boolean success
        +String notes
        +String branch
        +MigrationReport withBranch(String branch)
    }

    class MigrationsReport {
        +List~MigrationReport~ migration
    }

    class SoftwareProject {
        <<external>>
    }

    Domain --> MigrationTask : contains
    Domain --> MigrationPossibility : contains
    Domain --> MigrationPoint : contains
    Domain --> MigrationPoints : contains
    Domain --> MigrationReport : contains
    Domain --> MigrationsReport : contains

    MigrationTask --> Classification : has many
    MigrationTask --> SoftwareProject : creates
    MigrationPoints --> MigrationPoint : contains many
    MigrationReport --> MigrationPoint : references
    MigrationsReport --> MigrationReport : contains many
```

#### 4. **Git Safety Net**

Every change happens in its own branch:

- ✅ Successful changes are preserved
- ❌ Failed changes are automatically rolled back
- 🔄 No risk to your main codebase

## Step 4: Running Your First Modernization

### Start the Agent Shell

```bash
# Make the shell script executable
chmod +x scripts/shell.sh

# Start the Embabel shell
./scripts/shell.sh
```

You should see the Embabel shell prompt:

```
  ______ __  __ ____          ____  ______ _
 |  ____|  \/  |  _ \   /\   |  _ \|  ____| |
 | |__  | \  / | |_) | /  \  | |_) | |__  | |
 |  __| | |\/| |  _ < / /\ \ |  _ <|  __| | |
 | |____| |  | | |_) / ____ \| |_) | |____| |____
 |______|_|  |_|____/_/    \_\____/|______|______|

shell:>
```

### Configure Your Target Project

Before running the agent, you need to update the project path in `ModernizerAgent.java`:

```java
private SoftwareProject softwareProject = new SoftwareProject(
    "/path/to/your/legacy/java/project"  // Update this path
);
```

And update the Git repository path:

```java
this.repository = builder
    .setGitDir(Path.of(
        "/path/to/your/legacy/java/project",  // Update this path
        ".git").toFile())
    .build();
```

### Run the Modernization

In the Embabel shell, execute:

```bash
# Analyze and modernize legacy persistence code
x "Focus on modernizing legacy persistence patterns and deprecated Spring annotations"
```

### What Happens Next

The agent will:

1. **🔍 Scan** your codebase for modernization opportunities
2. **📝 Identify** specific files and issues
3. **🌿 Create** Git branches for each fix
4. **🤖 Generate** AI-powered solutions
5. **✅ Test** each change by building the project
6. **📊 Report** results with branch information

## Step 5: Understanding the Output

### Migration Points Report

The agent first generates a report of found issues:

```json
{
  "migrationPoints": [
    {
      "filePath": "src/main/java/com/example/UserRepository.java",
      "description": "Using deprecated @Repository annotation pattern",
      "classificationName": "Deprecated"
    },
    {
      "filePath": "src/main/java/com/example/SecurityConfig.java",
      "description": "Legacy WebSecurityConfigurerAdapter usage",
      "classificationName": "Security"
    }
  ],
  "overview": "Found 2 modernization opportunities focusing on Spring Boot 3.x compatibility"
}
```

### Migration Results

After processing, you'll get detailed results:

```json
{
  "migration": [
    {
      "migrationPoint": {...},
      "success": true,
      "notes": "Successfully updated to Spring Boot 3.x security configuration",
      "branch": "abc123-embabel-mod-1"
    }
  ]
}
```

## Step 6: Reviewing Changes

### Check Created Branches

```bash
# List all branches created by the agent
git branch | grep embabel-mod

# Review a specific change
git checkout abc123-embabel-mod-1
git diff main
```

### Merge Successful Changes

```bash
# For each successful branch
git checkout main
git merge abc123-embabel-mod-1
git branch -d abc123-embabel-mod-1
```

## Step 7: Customization

### Adding Custom Classifications

Edit `ModernizerAgent.java` to add your own migration categories:

```java
private final List<Classification> classifications = List.of(
    new Classification("Legacy", "Code that is outdated and may not follow current best practices."),
    new Classification("Deprecated", "Code that uses deprecated libraries or frameworks."),
    new Classification("Persistence", "Code related to persistence usage"),
    new Classification("Security", "Code related to security"),
    // Add your custom classifications
    new Classification("Performance", "Code with performance optimization opportunities"),
    new Classification("Testing", "Test code that needs modernization")
);
```

### Customizing Prompts

Modify the Jinja2 templates in `src/main/resources/prompts/`:

#### `find_migration_points.jinja`

```jinja
Identify code modernization points in the following project.

You are targeting migration from Java 8 to Java 17.
Focus specifically on: {{ focus }}

# Custom instructions for your organization
- Prioritize security-related changes
- Ensure compatibility with Spring Boot 3.5.6
- Follow company coding standards

{% for c in classifications %}
    - {{ c.name() }}: {{ c.description() }}
{% endfor %}
```

### Adding Custom Tools

Extend the agent with additional tools:

```java
@Action
public Domain.MigrationPoints migrationPoints(
        UserInput userInput,
        Ai ai) throws Exception {
    return ai
        .withAutoLlm()
        .withReferences(softwareProject)
        .withToolObject(new BashTools(softwareProject.getRoot()))
        .withToolObject(new CustomAnalysisTools())  // Add custom tools
        .withTemplate("find_migration_points")
        .createObject(Domain.MigrationPoints.class, Map.of(
            "focus", userInput.getContent(),
            "classifications", classifications
        ));
}
```

## Step 8: Advanced Usage

### Batch Processing Multiple Projects

Create a script to process multiple projects:

```bash
#!/bin/bash
projects=(
    "/path/to/project1"
    "/path/to/project2"
    "/path/to/project3"
)

for project in "${projects[@]}"; do
    echo "Processing $project..."
    # Update ModernizerAgent configuration
    # Run modernization
    # Collect results
done
```

### Integration with CI/CD

Add the agent to your CI pipeline:

```yaml
# .github/workflows/modernization.yml
name: Code Modernization
on:
  schedule:
    - cron: "0 2 * * 1" # Weekly on Monday

jobs:
  modernize:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - name: Setup Java
        uses: actions/setup-java@v3
        with:
          java-version: "21"
      - name: Run Modernization Agent
        run: |
          ./scripts/shell.sh -c "x 'Weekly modernization scan'"
      - name: Create Pull Request
        # Create PR with modernization results
```

## Troubleshooting

### Common Issues and Solutions

#### 1. **"Repository has no commits yet"**

```bash
# Initialize your target repository
cd /path/to/your/project
git init
git add .
git commit -m "Initial commit"
```

#### 2. **"Failed to connect to LLM provider"**

- Verify your API keys are set correctly
- Check network connectivity
- Ensure you have sufficient API credits

#### 3. **"Build failures after modernization"**

- Review the generated changes manually
- Check for missing dependencies
- Verify Java version compatibility

#### 4. **"Agent not finding migration points"**

- Ensure your project path is correct
- Check that the project contains Java files
- Verify the focus prompt is specific enough

### Debug Mode

Enable detailed logging:

```properties
logging.level.com.embabel.modernizer=DEBUG
logging.level.com.embabel.agent=DEBUG
```

## Best Practices

### 1. **Start Small**

Begin with a small, non-critical project to understand the agent's behavior.

### 2. **Review All Changes**

Always review generated code before merging to main branch.

### 3. **Backup Your Code**

Ensure you have backups before running large-scale modernizations.

### 4. **Test Thoroughly**

Run your full test suite after applying modernizations.

### 5. **Iterative Approach**

Process one classification type at a time for better control.

## Next Steps

Now that you've mastered the basics, consider:

1. **Extending the Agent**: Add support for other languages or frameworks
2. **Custom Rules**: Implement organization-specific modernization rules
3. **Metrics Collection**: Track modernization progress across projects
4. **Team Integration**: Set up shared configurations for your development team

## Conclusion

Congratulations! You've successfully set up and used the Java Code Modernizer Agent. This powerful tool can save countless hours of manual refactoring while ensuring your codebase stays current with modern Java and Spring Boot practices.

The combination of AI-powered analysis, safe Git branching, and automated testing makes this agent a valuable addition to any Java development workflow.

## Resources

- [Embabel Framework Documentation](https://github.com/embabel/embabel-agent)
- [Spring AI Documentation](https://docs.spring.io/spring-ai/reference/)
- [Java 17 Migration Guide](https://docs.oracle.com/en/java/javase/17/migrate/)
- [Spring Boot 3.x Migration Guide](https://github.com/spring-projects/spring-boot/wiki/Spring-Boot-3.0-Migration-Guide)

---

_Happy modernizing! 🚀_
