# Quarkus-based Financial Entry Management System

This project is a Quarkus-based RESTful application for managing financial entries with DynamoDB integration and AWS Lambda deployment capabilities.

The system provides a comprehensive set of APIs for creating, retrieving, and managing financial entries across multiple accounts. It leverages Quarkus for rapid development and AWS services for scalable, serverless deployment.

Key features include:
- Account-based entry management
- Category-based filtering and updating
- Date range querying
- Balance calculation
- DynamoDB integration for data persistence
- AWS Lambda deployment support

## Repository Structure

```
.
├── docker-compose.yml
├── pom.xml
├── README.md
├── src
│   ├── main
│   │   └── java
│   │       └── academy
│   │           └── silicon
│   │               ├── GreetingResource.java
│   │               ├── model
│   │               │   └── Entry.java
│   │               ├── resource
│   │               │   └── EntryResource.java
│   │               └── service
│   │                   ├── EntryHelper.java
│   │                   └── EntryService.java
│   └── test
│       └── java
│           └── academy
│               └── silicon
│                   ├── GreetingResourceIT.java
│                   └── GreetingResourceTest.java
└── template.yml
```

Key Files:
- `pom.xml`: Maven project configuration file
- `src/main/java/academy/silicon/resource/EntryResource.java`: Main REST resource for entry management
- `src/main/java/academy/silicon/service/EntryService.java`: Service layer for business logic
- `docker-compose.yml`: Docker configuration for local DynamoDB instance
- `template.yml`: AWS SAM template for Lambda deployment

## Usage Instructions

### Installation

Prerequisites:
- Java 17
- Maven 3.8+
- Docker (for local development with DynamoDB)

Steps:
1. Clone the repository
2. Navigate to the project root directory
3. Run `mvn clean install` to build the project

### Getting Started

To run the application locally:

1. Start the local DynamoDB instance:
   ```
   docker-compose up -d
   ```

2. Run the Quarkus application in dev mode:
   ```
   mvn quarkus:dev
   ```

The application will be available at `http://localhost:8080`.

### Configuration

The application uses the following environment variables:
- `QUARKUS_DYNAMODB_ENDPOINT_OVERRIDE`: DynamoDB endpoint (default: `http://localhost:8000` for local development)
- `AWS_REGION`: AWS region for DynamoDB (default: `us-east-1`)

### Common Use Cases

1. Add a new entry:
   ```
   POST /entryResource
   {
     "accountID": "123",
     "description": "Grocery shopping",
     "amount": 50.00,
     "date": "2023-04-15+10:30",
     "category": "Food"
   }
   ```

2. Retrieve entries for an account:
   ```
   GET /entryResource/findByAccount/123
   ```

3. Find entries by category:
   ```
   GET /entryResource/findByCategory/Food
   ```

4. Replace category for entries:
   ```
   GET /entryResource/replaceCategory/OldCategory/NewCategory
   ```

### Testing & Quality

Run unit tests:
```
mvn test
```

Run integration tests:
```
mvn verify -Pnative
```

### Troubleshooting

Common Issue: DynamoDB connection fails

Problem: The application cannot connect to the local DynamoDB instance.

Error message: "Unable to connect to DynamoDB at http://localhost:8000"

Diagnostic steps:
1. Ensure Docker is running: `docker ps`
2. Check if the DynamoDB container is up: `docker-compose ps`
3. Verify the DynamoDB port is exposed: `docker-compose logs dynamodb-local`

Solution:
- If the container is not running, start it with `docker-compose up -d`
- If the port is not exposed, check the `docker-compose.yml` file for correct port mapping

Expected outcome: The application should connect to DynamoDB successfully after resolving the issue.

### Debugging

To enable debug mode:
1. Set the `QUARKUS_LOG_LEVEL` environment variable to `DEBUG`
2. Restart the application

Debug output can be found in the console when running in dev mode, or in the CloudWatch logs when deployed to AWS Lambda.

Log file location: `target/quarkus.log`

### Performance Optimization

Key metrics to monitor:
- API response times
- DynamoDB read/write capacity units consumed

Use AWS X-Ray for tracing when deployed to AWS Lambda to identify performance bottlenecks.

## Data Flow

The application follows a typical RESTful architecture for handling financial entry data:

1. Client sends HTTP requests to the `EntryResource` endpoints.
2. `EntryResource` processes the request and delegates to `EntryService`.
3. `EntryService` performs business logic and interacts with DynamoDB using the AWS SDK.
4. Data is persisted in or retrieved from DynamoDB.
5. Results are returned through the service and resource layers back to the client.

```
Client <-> EntryResource <-> EntryService <-> DynamoDB
```

Important technical considerations:
- The application uses the Quarkus DynamoDB extension for efficient interaction with DynamoDB.
- Balance calculations are performed in-memory to reduce database load.
- Entries are sorted by date to ensure consistent balance calculations.

## Deployment

Prerequisites:
- AWS CLI configured with appropriate permissions
- SAM CLI installed

Deployment steps:
1. Build the application:
   ```
   mvn clean package
   ```

2. Deploy using SAM:
   ```
   sam deploy --guided
   ```

3. Follow the prompts to configure the deployment.

Environment configurations:
- Set the `QUARKUS_DYNAMODB_ENDPOINT_OVERRIDE` to the appropriate DynamoDB endpoint for your AWS environment.
- Ensure the Lambda function has the necessary IAM permissions to access DynamoDB.

## Infrastructure

The application uses the following AWS resources:

Lambda:
- `piggybankv2`: Main Lambda function handling HTTP API events
  - Runtime: Java 17
  - Memory: 1024 MB
  - Timeout: 15 seconds

DynamoDB:
- Table: `entry` (defined in the DynamoDB CRUD policy)

API Gateway:
- HTTP API integrated with the Lambda function

The infrastructure is defined using AWS SAM in the `template.yml` file, enabling easy deployment and management of the serverless application.